#!/usr/bin/env python
#
# xkmap_base.py   --  Basic framework for xkmap
#
# (c) Copyright 2009-2010 Michael Towers (larch42 at googlemail dot com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------
# 2010.08.08

# For xorg-1.8 (and above?)

import os, tempfile, re, threading
from subprocess import call

from xkmap_conf import *
from liblarch.rootrun import init_rootrun
from liblarch.translation import i18n_module, i18ndoc

# Get xkmap directory
thisdir = os.path.dirname(os.path.realpath(__file__))

import gettext
_ = i18n_module(thisdir, 'xkmap')


class Xkmap:
    def __init__(self, chroot=''):
        """The chroot parameter allows the module to work on mounted
        systems, rather than the currently running one.
        """
        self.chroot = chroot
        self.home = os.environ['HOME']

        self.models, self.layouts, self.allVariants = self.getLayouts()

        # Get initial layout
        self.config0 = self.getglobal()
        userdata = self.getuser()
        if userdata:
            self.config0 = userdata


    def fss_fetch_layout(self, fname):
        fh = open(thisdir + '/' + fname)
        r = fh.read()
        fh.close()
        return eval(r)


    def fss_about(self):
        return i18ndoc(thisdir + '/i18n/help/About.html') % {
                'xconf': configdir + '/' + configfile,
                'xrules': base_lst,
                'uset': configuser,
                'xkmconf': thisdir + '/xkmap_conf.py',
                'xmodmapg': xmodmapg,
                'xmodmapu': self.home + '/' + xmodmapu,
                'xmodmap': thisdir + '/xmodmap'
            }


    def fss_models(self):
        return self.models


    def fss_layouts(self):
        return self.layouts


    def fss_startdata(self):
        if self.config0:
            return self.config0
        else:
            return (DEFAULTMODEL, DEFAULTLAYOUT, STANDARDVARIANT[0])


    def fss_variantlist(self, layout):
        vl = self.allVariants.get(layout)
        return vl if vl else [STANDARDVARIANT]


    def fss_uenabled(self):
        return self.userenabled


    def fss_command(self, m, l, v):
        cmd = 'setxkbmap %s %s' % ((modelopt % m), (layoutopt % l))
        if (v != STANDARDVARIANT[0]):
            cmd += variantopt % v
        return cmd


    def getLayouts(self):
        """Read 'base.lst' to get lists of models, layouts and variants
        """
        blf = None
        try:
            blf = open(self.chroot + base_lst)
            while blf.readline().strip() != '! model': pass

            models = []
            while True:
                line = blf.readline().strip()
                if line:
                    if line == '! layout': break
                    models.append(line.split(None, 1))

            layouts = []
            allVariants = {}
            while True:
                line = blf.readline().strip()
                if line:
                    if line == '! variant': break
                    lt = line.split(None, 1)
                    layouts.append(lt)
                    allVariants[lt[0]] = [STANDARDVARIANT]
            layouts.sort()

            while True:
                line = blf.readline().strip()
                if line:
                    if line == '! option': break
                    parts = line.split(None, 2)
                    line = (parts[0], parts[2])
                    layout = parts[1].rstrip (':')
                    allVariants[layout].append(line)

        finally:
            if blf != None:
                blf.close()
        return (models, layouts, allVariants)


    def parseconfig(self, filepath, extra=None):
        """Read the layout specified in an xkmap configuration, if it exists,
        as a triple, otherwise return 'None'.
        extra is a search regexp (multiline search), which, if it is found,
        will force a None result.
        """
        res = []
        if os.path.isfile(filepath):
            fh = open(filepath)
            data = fh.read()
            fh.close()
            for field in ('MODEL', 'LAYOUT', 'VARIANT'):
                m = re.search(r'^#%s:(.*)$' % field, data, re.MULTILINE)
                if m:
                    res.append(m.group(1))
                else:
                    return None
            if extra:
                return (None if re.search(extra, data, re.MULTILINE)
                        else res)
            else:
                return res
        else:
            return None


    def getglobal(self):
        """Read the layout specified in an existing 'configfile', or else
        return 'None'.
        """
        self.cfile = '%s/%s' % (configdir, configfile)
        return self.parseconfig(self.chroot + self.cfile)


    def getuser(self):
        if self.chroot:
            return None
        self.ufile = '%s/%s' % (self.home, configuser)
        res = self.parseconfig(self.ufile, r'^#X#')
        self.userenabled = bool(res)
        return res


    def fss_setnow(self, *args):
        if self.chroot:
            debug("setxkbmap is not supported with 'chrooting'")
            return (False, "")
        else:
            call(self.fss_command(*args), shell=True)
            if os.path.isfile(xmodmapg):
                call(['xmodmap', xmodmapg])
            mfile = '%s/%s' % (self.home, xmodmapu)
            if os.path.isfile(mfile):
                call(['xmodmap', mfile])

            return (True, _("Keymap set to:\n"
                    "  Model: %s\n  Layout: %s\n  Variant: %s")
                    % args)


    def fss_setuser(self, *args):
        if self.chroot:
            debug("User map setting is not supported with 'chrooting'")
            return (False, "")
        else:
            self.writeuserdata(args)
            return self.fss_setnow(*args)


    def fss_clearuser(self):
        if self.chroot:
            debug("User map clearing is not supported with 'chrooting'")
        elif self.userenabled:
            cfg = self.parseconfig(self.ufile)
            if not cfg:
                self.userenabled = False
                return (False, _("Couldn't parse user settings in '%s'")
                        % self.ufile)
            self.writeuserdata(cfg, False)
            return (True, _("User-specific settings disabled"))
        return (False, None)


    def writeuserdata(self, args, enable=True):
        text = template_user.replace('___M___', args[0])
        text = text.replace('___L___', args[1])
        text = text.replace('___V___', args[2])
        cmd = '' if enable else '#X#'
        cmd += self.fss_command(*args)
        text = re.sub(r'(?m)^#X#.*$', cmd, text)    # (?m) for 'multiline'

        udir = os.path.dirname(self.ufile)
        if not os.path.isdir(udir):
            os.makedirs(udir)
        fh = open(self.ufile, 'w')
        fh.write(text)
        fh.close()
        # Make the file executable
        call(['chmod', '755', self.ufile])
        self.userenabled = enable


    def fss_setglobal(self, m, l, v):
        self.mlv = (m, l, v)
        text = template_xorg.replace('___M___', m)
        text = text.replace('#***M***', modelLine % m)
        text = text.replace('___L___', l)
        text = text.replace('#***L***', layoutLine % l)
        text = text.replace('___V___', v)
        if (v != STANDARDVARIANT[0]):
            text = text.replace('#***V***', variantLine % v)

        udir = os.path.dirname(self.ufile)
        if not os.path.isdir(self.chroot + configdir):
            return (False, _("Xorg configuration directory (%s) does not exist.\n"
                    "Is Xorg>=1.8 installed?") % configdir)
        path = self.chroot + self.cfile
        # Save the text to a root owned file at 'path'.
        fh, self.temp_path = tempfile.mkstemp()
        f = os.fdopen(fh, 'w')
        f.write(text)
        f.close()

        cmd = ("mv '%s' '%s' && chmod 644 '%s' && chown root:root '%s'"
                % (self.temp_path, path, path, path))
        self.rcall = init_rootrun(self._pwget)
        self.rcall.run(cmd, self._rootsave_done)
        return None

    def _pwget(self, cb, prompt):
        self.pw_event = threading.Event()
        ui_signal('get_password', prompt)
        self.pw_event.wait()
        self.rcall.cb_done(cb, *self.pw_returned)

    def fss_sendpassword(self, ok, pw):
        # The result needs to be passed to the waiting _pwget method
        # (which is running in a different thread).
        self.pw_returned = (ok, pw)
        self.pw_event.set()
        return None

    def _rootsave_done(self, cb, ok, op):
        if ok:
            ui_signal('showcompleted', *self.fss_setnow(*self.mlv))
        else:
            os.remove(self.temp_path)
            ui_signal('showcompleted', False, op)
